/*
 * Copyright (C) 2000 - 2024 Silverpeas
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * As a special exception to the terms and conditions of version 3.0 of
 * the GPL, you may redistribute this Program in connection with Free/Libre
 * Open Source Software ("FLOSS") applications as described in Silverpeas's
 * FLOSS exception. You should have received a copy of the text describing
 * the FLOSS exception, and it is also available here:
 * "https://www.silverpeas.org/legal/floss_exception.html"
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.silverpeas.core.cmis.model;

import org.apache.chemistry.opencmis.commons.enums.Action;
import org.apache.chemistry.opencmis.commons.enums.BaseTypeId;
import org.silverpeas.core.contribution.model.ContributionIdentifier;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;

/**
 * A publication in Silverpeas. A publication is a contribution of a user that gathers one or more
 * contents on a similar topic. Such content can be a WYSIWYG text or a form, and one or more
 * attached documents.
 * <p>
 * A publication, in our CMIS implementation, is contained either into a contribution folder (for
 * applications categorizing their resources) or directly at the root level of the application. For
 * applications using the {@link ContributionFolder}s to categorize the publications, usually a root
 * virtual folder is used to represent the application itself; in that case, for publications
 * organized into such a folder, their parent is then considered to be the application itself.
 * </p>
 *
 * @author mmoquillon
 */
public class Publication extends CmisFolder {

  /**
   * The identifier of its CMIS type.
   */
  public static final TypeId CMIS_TYPE = TypeId.SILVERPEAS_PUBLICATION;
  /**
   * The UTF-8 symbol representing the type of this CMIS object. It acts as an icon but instead
   * of being a regular image the icon is encoded in UTF-8.
   */
  public static final String SYMBOL = "\uD83D\uDCD4";

  private final ContributionIdentifier id;

  public static List<TypeId> getAllAllowedChildrenTypes() {
    return Collections.singletonList(TypeId.SILVERPEAS_DOCUMENT);
  }

  /**
   * Constructs a new publication with the specified identifier, name and language.
   *
   * @param id       the {@link ContributionIdentifier} instance identifying the publication in
   *                 Silverpeas.
   * @param name     the name of the publication.
   * @param language the language in which is written the publication.
   */
  Publication(final ContributionIdentifier id, final String name, final String language) {
    super(id, name, language);
    this.id = id;
  }

  public String getApplicationId() {
    return id.getComponentInstanceId();
  }

  @Override
  public String getSymbol() {
    return SYMBOL;
  }

  @Override
  public boolean isRoot() {
    return false;
  }

  @Override
  public List<TypeId> getAllowedChildrenTypes() {
    return getAllAllowedChildrenTypes();
  }

  @Override
  public BaseTypeId getBaseTypeId() {
    return BaseTypeId.CMIS_FOLDER;
  }

  @Override
  public TypeId getTypeId() {
    return CMIS_TYPE;
  }

  @Override
  protected Supplier<Set<Action>> getAllowableActionsSupplier() {
    return () -> completeWithPublicationActions(completeWithFolderActions(theCommonActions()));
  }

  private Set<Action> completeWithPublicationActions(final Set<Action> actions) {
    actions.add(Action.CAN_CREATE_DOCUMENT);
    return actions;
  }
}
  